/***
*putwch.c - write a wide character to console
*
*       Copyright (c) Microsoft Corporation. All rights reserved.
*
*Purpose:
*       defines _putwch() - writes a wide character to a console
*
*******************************************************************************/


#include <stdlib.h>
#include <conio.h>
#include <io.h>
#include <errno.h>
#include <cruntime.h>
#include <stdio.h>
#include <dbgint.h>
#include <file2.h>
#include <internal.h>
#include <mtdll.h>
#include <limits.h>

/*
 * declaration for console handle
 */
extern intptr_t _confh;

/***
*wint_t _putwch(ch) - write a wide character to a console
*
*Purpose:
*       Writes a wide character to a console.
*
*Entry:
*       wchar_t ch - wide character to write
*
*Exit:
*       returns the wide character if successful
*       returns WEOF if fails
*
*Exceptions:
*
*******************************************************************************/

wint_t _CRTIMP __cdecl _putwch (
        wchar_t ch
        )
{
        REG2 wint_t retval;

        _mlock(_CONIO_LOCK);
        __try {

        retval = _putwch_nolock(ch);

        }
        __finally {
                _munlock(_CONIO_LOCK);
        }

        return(retval);
}

/***
*_putwch_nolock() -  _putwch() core routine (locked version)
*
*Purpose:
*       Core _putwch() routine; assumes stream is already locked.
*
*       [See _putwch() above for more info.]
*
*Entry: [See _putwch()]
*
*Exit:  [See _putwch()]
*
*Exceptions:
*
*******************************************************************************/

wint_t __cdecl _putwch_nolock (
        wchar_t ch
        )
{

    int size, num_written;
    static int use_w = 2;
    char mbc[MB_LEN_MAX +1];
    if ( use_w)
    {
        if (_confh == -2)
            __initconout();

        /* write character to console file handle */

        if (_confh == -1)
            return WEOF;
        else if ( !WriteConsoleW( (HANDLE)_confh,
                                  (LPVOID)&ch,
                                  1,
                                  &num_written,
                                  NULL )
                  )
        {
            if ( use_w == 2 && GetLastError() == ERROR_CALL_NOT_IMPLEMENTED)
                use_w = 0;
            else
                return WEOF;
        } else
                use_w = 1;
    }

    if ( use_w == 0)
    {
        size = WideCharToMultiByte(
                                   GetConsoleOutputCP(),
                                   0,
                                   (LPWSTR)&ch, 1,
                                   mbc,
                                   MB_LEN_MAX,
                                   NULL,
                                   NULL
                                   );
        if ( (_confh == -1) || !WriteConsole( (HANDLE)_confh,
                                              (LPVOID)mbc,
                                              size,
                                              &num_written,
                                              NULL )
           )
                /* return error indicator */
                return WEOF;
    }
    return ch;
}

/***
*  _cputws() - _cputws() writes a wide char string to console.
*
*  Purpose:
*       Writes a wide char string to console.
*
*  Entry:
*       str:    pointer to string
*  Exit:
*       returns 0 if sucessful. Nonzero if unsucessful
*
*******************************************************************************/
int _CRTIMP __cdecl _cputws(
        const wchar_t *str
        )
{
    size_t len;
    int retval = 0;

    _VALIDATE_CLEAR_OSSERR_RETURN((str != NULL), EINVAL, -1);

    len = wcslen(str);
    _mlock(_CONIO_LOCK);
    __try {
    while(len--)
    {
        if ( _putwch_nolock(*str++) == WEOF)
        {
            retval = -1;
            break;
        }
    }
    }
    __finally {
            _munlock(_CONIO_LOCK);
    }
    return retval;
}

